import { __ } from "@wordpress/i18n";
import {
	RichText,
	useBlockProps,
	InspectorControls,
} from "@wordpress/block-editor";
import {
	Button,
	TabPanel,
	PanelBody,
	ToggleControl,
	Panel,
	PanelRow,
} from "@wordpress/components";
import { useState } from "@wordpress/element";
import { useSelect } from "@wordpress/data";

import "./editor.scss";
import preview from './assets/accordion-tabs.jpg';

export default function Edit({ attributes, setAttributes }) {
	const { heading, tabs, firstOpen } = attributes;
	const [tabIndex, setTabIndex] = useState(tabs.length);
	const [tabId, setTabId] = useState(tabs.length);
	const { deviceType } = useSelect(
		(select) => ({
			deviceType: select("core/edit-post").__experimentalGetPreviewDeviceType(),
		}),
		[],
	);

	const scapeToEvent = (e) => {
		e.preventDefault();
	};

	const addTab = () => {
		const newTab = [...tabs];
		const tab = {
			name: `tab-${tabIndex}`,
			title: `New tab ${tabIndex}`,
			faqs: [],
		};
		setTabIndex(tabIndex + 1);
		newTab.push(tab);
		setAttributes({
			tabs: newTab,
		});
	};

	const removeTab = () => {
		const currentTabs = [...tabs];
		const updatedTabs = currentTabs.filter((tab, index) => index !== tabId);
		const updatedTabsWithNewNames = updatedTabs.map((tab, index) => ({
			...tab,
			name: `tab-${index}`,
		}));

		setAttributes({
			tabs: updatedTabsWithNewNames,
		});
	};

	const addElement = () => {
		const newElement = [...tabs];
		const element = {
			title: "Title",
			faq: "Content",
		};

		newElement[tabId].faqs.push(element);
		setAttributes({
			tabs: newElement,
		});
	};

	const removeElement = (elm) => {
		const elements = [...tabs];

		elements[tabId].faqs.splice(elm, 1);
		setAttributes({
			tabs: elements,
		});
	};

	const updateTabTitle = (newTitle) => {
		const updatedTabs = [...tabs];

		updatedTabs[tabId].title = newTitle;
		setAttributes({ tabs: updatedTabs });
	};

	const getTabId = (tab) => {
		const parts = tab.split("-");

		if (parts.length === 2) {
			try {
				const tabId = parseInt(parts[1], 10);
				if (!isNaN(tabId)) {
					setTabId(tabId);
				}
			} catch (error) {
				console.warn(error);
			}
		}
	};

	const updateElement = (value, parentIndex, index, type) => {
		const updateElement = [...tabs];
		getTabId(parentIndex);

		updateElement[tabId].faqs[index][type] = value;
		setAttributes({ tabs: updateElement });
	};
	return (
		<>
			<InspectorControls key="setting">
				<Panel className="components-panel--custom">
					<PanelRow>
						<ToggleControl
							label="First accordion open on default?"
							help={
								firstOpen ? "Open on default" : "Closed on default"
							}
							checked={firstOpen}
							onChange={(firstOpen) => {
								setAttributes({ firstOpen });
							}}
						/>
					</PanelRow>
				</Panel>
			</InspectorControls>
			<section
				{...useBlockProps()}
				className={`container accordion-tabs block-editor-block-list__block wp-block ${deviceType}`}
				data-comp="accordion-tabs" data-preview="true"
			>

				<figure className="preview">
					<img src={preview} alt="" />
				</figure>

				<div className="mobile-text">
					<h3>Edit this block in Desktop</h3>
				</div>
				<header className="row no-mobile">
					<RichText
						tagName="h2"
						value={heading}
						className="col-12-12"
						onChange={(newHeading) => setAttributes({ heading: newHeading })}
						placeholder="Heading"
					/>
				</header>
				<div className="row">
					<section className="col-12-12">
						<TabPanel
							key="tabPanel"
							className="tab-panel"
							activeClass="active-tab"
							orientation="horizontal"
							initialTabName="tab-0"
							onSelect={(tabName) => getTabId(tabName)}
							tabs={tabs}
						>
							{(tab) => (
								<PanelBody key={tab.name}>
									<div className="tab-panel__header">
										<RichText
											tagName="h3"
											value={tab.title}
											onChange={(val) => updateTabTitle(val, tab.name)}
											placeholder="H3 TITLE"
										/>
										<Button
											className="bttn--danger"
											label="Remove"
											onClick={(e) => {
												removeTab();
											}}
										>
											Remove Tab
										</Button>
									</div>
									{tab.faqs.map((value, i) => (
										<details key={i}>
											<summary>
												<RichText
													value={value.title}
													onChange={(val) =>
														updateElement(val, tab.name, i, "title")
													}
													placeholder="TITLE"
													onClick={scapeToEvent}
												/>
												<Button
													label="Remove"
													onClick={(e) => {
														removeElement(i);
													}}
													className="bttn--danger"
												>
													Remove Item
												</Button>
											</summary>
											<RichText
												value={value.faq}
												onChange={(val) =>
													updateElement(val, tab.name, i, "faq")
												}
												placeholder="content"
											/>
										</details>
									))}
									<Button
										variant="secundary"
										onClick={(e) => {
											addElement(tab.name);
										}}
										className="bttn--secondary"
									>
										Add item
									</Button>
								</PanelBody>
							)}
						</TabPanel>
						<Button
							variant="primary"
							onClick={(e) => {
								addTab("New Tab");
							}}
						>
							Add new Tab
						</Button>
					</section>
				</div>
			</section>
		</>
	);
}
