/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText } from "@wordpress/block-editor";

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save(props) {
	const { attributes } = props;

	return (
		<section
			{...useBlockProps.save()}
			className={`accordion-tabs ${attributes.firstOpen ? "first-open" : ""}`}
			data-comp="accordion-tabs"
		>
			<header className="row no-mobile">
				{attributes.heading && (
					<RichText.Content
						tagName="h2"
						className="col-12-12"
						value={attributes.heading}
					/>
				)}
			</header>
			<menu className="row">
				<ul className="tabs no-mobile col-12-12">
					{attributes.tabs.map((tab, i) => (
						<li key={i}>
							{
								<RichText.Content
									tagName="button"
									className="tab"
									data-tab={i}
									value={tab.title}
								/>
							}
						</li>
					))}
				</ul>
			</menu>
			{attributes.tabs.map((tab, i) => (
				<div className="row group" data-tab={i} key={i}>
					<header>
						{tab.title && (
							<RichText.Content
								tagName="h3"
								className="mobile"
								value={tab.title}
							/>
						)}
					</header>
					{tab.faqs.map((faq, i) => (
						<article
							className={`col-12-12 accordion`}
							key={i}
						>
							<header>
								{faq.title && (
									<RichText.Content tagName="h4" value={faq.title} />
								)}
								<div>
									<i class="icon-arrow-left"></i>
								</div>
							</header>
							<div className="content-wrapper">
								{faq.faq && <RichText.Content tagName="p" value={faq.faq} />}
							</div>
						</article>
					))}
				</div>
			))}
		</section>
	);
}
