/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from "@wordpress/i18n";

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	RichText,
	useBlockProps,
	InspectorControls,
} from "@wordpress/block-editor";
import {
	Button,
	TabPanel,
	PanelBody,
	ToggleControl,
	Panel,
	PanelRow,
} from "@wordpress/components";

import { useState } from "@wordpress/element";

import preview from './assets/accordion-two-columns.jpg';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import "./editor.scss";

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes }) {
	const { title, subtitle, tabs, firstOpen } = attributes;
	const [elmIndex, setElmIndex] = useState(tabs.length);
	const [currentEdit, setCurrentEdit] = useState(false);

	const updateTabValue = (value, attr, tabId) => {
		const updatedTabs = [...tabs];

		updatedTabs[tabId][attr] = value;
		setAttributes({ tabs: updatedTabs });
	};

	const updateAsideValue = (value, tabId, attr, key) => {
		const updatedTabs = [...tabs];

		updatedTabs[tabId].aside[key][attr] = value;
		setAttributes({ tabs: updatedTabs });
	};

	const addElement = () => {
		const newElement = [...tabs];
		const elm = {
			name: ``,
			overview: ``,
			description: "",
			asidetitle: "",
			aside: [],
		};
		setElmIndex(elmIndex + 1);
		newElement.push(elm);
		setAttributes({
			tabs: newElement,
		});
	};

	const addAsideElement = (tabId) => {
		const newElement = [...tabs];
		const elm = {
			title: ``,
			content: ``,
		};
		newElement[tabId].aside.push(elm);
		setAttributes({
			tabs: newElement,
		});
	};

	const removeElement = (elm) => {
		const elements = [...tabs];

		elements.splice(elm, 1);
		setAttributes({
			tabs: elements,
		});
	};

	const removeAsideElement = (tab, elm) => {
		const elements = [...tabs];

		elements[tab].aside.splice(elm, 1);
		setAttributes({
			tabs: elements,
		});
	};

	let tab;

	if (currentEdit === 0) {
		tab = tabs[0];
	} else if (currentEdit !== false) {
		tab = tabs[currentEdit];
	}

	let tabContent = null;

	if (tab) {
		tabContent = (
			<article className="accordion" key={currentEdit} data-tab={currentEdit}>
				<header className="mobile">
					<h4>{tab.name}</h4>
					<div>
						<i className="icon"></i>
					</div>
				</header>
				<div className="content-wrapper">
					<div className="row-wrapper">
						<div className="main">
							<RichText
								tagName="h5"
								value={tab.overview}
								onChange={(newValue) =>
									updateTabValue(newValue, "overview", currentEdit)
								}
								placeholder="Overview..."
							/>
							<RichText
								tagName="p"
								value={tab.description}
								onChange={(newValue) =>
									updateTabValue(newValue, "description", currentEdit)
								}
								placeholder="Description..."
							/>
						</div>
						<aside>
							<RichText
								tagName="h6"
								value={tab.asidetitle}
								onChange={(newValue) =>
									updateTabValue(newValue, "asidetitle", currentEdit)
								}
								placeholder="Title..."
							/>
							<ul>
								{tab.aside.map((elm, k) => (
									<li key={k}>
										<RichText
											tagName="strong"
											value={elm.title}
											onChange={(newValue) =>
												updateAsideValue(newValue, currentEdit, "title", k)
											}
											placeholder="Title..."
										/>
										<RichText
											tagName="p"
											value={elm.content}
											onChange={(newValue) =>
												updateAsideValue(newValue, currentEdit, "content", k)
											}
											placeholder="Content..."
										/>
                                        
										<Button
											variant="primary"
											isDestructive
											onClick={(e) => {
												removeAsideElement(currentEdit, k);
											}}
										>
											-
										</Button>
									</li>
								))}
							</ul>
							<Button
								variant="primary"
								onClick={(e) => {
									addAsideElement(currentEdit);
								}}
							>
								Add Element
							</Button>
						</aside>
					</div>
				</div>
			</article>
		);
	}

	return (
		<>
			<InspectorControls key="setting">
				<Panel className="components-panel--custom">
					<PanelRow>
						<ToggleControl
							label="First accordion open on default?"
							help={firstOpen ? "Open on default" : "Closed on default"}
							checked={firstOpen}
							onChange={(firstOpen) => {
								setAttributes({ firstOpen });
							}}
						/>
					</PanelRow>
				</Panel>
			</InspectorControls>
			<section
				{...useBlockProps()}
				className="accordion-two-columns block-editor-block-list__block wp-block"
				data-comp="accordion-two-columns" data-preview="true"
			>

				<figure className="preview">
					<img src={preview} alt="" />
				</figure>

				<div className="row top">
					<header className="offset-left-12-1 col-12-10">
						<RichText
							tagName="h2"
							value={title}
							onChange={(newValue) => setAttributes({ title: newValue })}
							placeholder="Title..."
						/>
						<RichText
							tagName="h3"
							value={subtitle}
							className="no-mobile"
							onChange={(newValue) => setAttributes({ subtitle: newValue })}
							placeholder="Subtitle..."
						/>
					</header>
					<menu className="offset-left-12-1 col-12-5">
						<ul className="tabs no-mobile">
							{tabs.map((tab, i) => (
								<li
									className="tab"
									key={i}
									data-tab={i}
									role="button"
									aria-label="accordion toggle"
								>
									<header>
										<RichText
											tagName="h4"
											value={tab.name}
											onChange={(newValue) =>
												updateTabValue(newValue, "name", i)
											}
											placeholder="Title..."
										/>
										<div>
											<Button
												variant="primary"
												isDestructive
												onClick={(e) => {
													removeElement(i);
												}}
											>
												Remove
											</Button>
											<Button
												variant="primary"
												onClick={(e) => {
													setCurrentEdit(i);
												}}
											>
												Edit
											</Button>
										</div>
									</header>
								</li>
							))}
						</ul>
					</menu>
					<div className="aside-content col-12-6">{tabContent}</div>
				</div>
				<div className="offset-left-12-1 col-12-10">
					<Button
						variant="primary"
						onClick={(e) => {
							addElement();
						}}
					>
						Add Element
					</Button>
				</div>
			</section>
		</>
	);
}
