/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText } from "@wordpress/block-editor";

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save(props) {
	const { attributes } = props;

	return (
		<section
			{...useBlockProps.save()}
			className={`accordion-two-columns ${attributes.firstOpen ? "first-open" : ""}`}
			data-comp="accordion-two-columns"
		>
			<div className="row top">
				<header className="offset-left-12-1 col-12-10">
					{attributes.title && (
						<RichText.Content
							tagName="h2"
							value={attributes.title}
							className="no-mobile"
						/>
					)}
					{attributes.subtitle && (
						<RichText.Content
							tagName="h3"
							value={attributes.subtitle}
							className="no-mobile"
						/>
					)}
				</header>
				<menu className="offset-left-12-1 col-12-5">
					<ul className="tabs no-mobile">
						{attributes.tabs.map((tab, k) => (
							<li
								className="tab"
								data-tab={k}
								key={k}
								role="button"
								aria-label="accordion toggle"
							>
								<header>
									{tab.name && (
										<RichText.Content
											tagName="h4"
											value={tab.name}
											className="no-mobile"
										/>
									)}
									<div>
										<i className="icon"></i>
									</div>
								</header>
							</li>
						))}
					</ul>
				</menu>
				<div className="aside-content col-12-6">
					{attributes.tabs.map((tab, k) => (
						<article className="accordion" key={k} data-tab={k}>
							<header className="mobile">
								{tab.name && <RichText.Content tagName="h4" value={tab.name} />}
								<div>
									<i className="icon"></i>
								</div>
							</header>
							<div className="content-wrapper">
								<div className="row-wrapper">
									<div className="main">
										{tab.overview && (
											<RichText.Content tagName="h5" value={tab.overview} />
										)}
										{tab.description && (
											<RichText.Content tagName="p" value={tab.description} />
										)}
									</div>
									<aside>
										{tab.asidetitle && (
											<RichText.Content tagName="h6" value={tab.asidetitle} />
										)}
										<ul>
											{tab.aside.map((elm, i) => (
												<li key={i}>
													{elm.title && (
														<RichText.Content
															tagName="strong"
															value={elm.title}
														/>
													)}
													{elm.content && (
														<RichText.Content tagName="p" value={elm.content} />
													)}
												</li>
											))}
										</ul>
									</aside>
								</div>
							</div>
						</article>
					))}
				</div>
			</div>
		</section>
	);
}
