import { __ } from "@wordpress/i18n";
import { MediaUpload } from "@wordpress/block-editor";
import {
	__experimentalDivider as Divider,
	Button,
	PanelBody,
	PanelRow,
	ToggleControl,
} from "@wordpress/components";
import { useState, useEffect } from "react";

const RenderImageControl = ({
	render,
	image,
	setAttributes,
	attribute,
	name,
}) => {
	return render ? (
		<>
			<PanelRow>
				<h2>Image Desktop</h2>
			</PanelRow>
			<PanelRow>
				<MediaUpload
					allowedTypes={["image"]}
					onSelect={(media) => {
						setAttributes({
							[name]: {
								...attribute,
								image: { ...image, desktop: media.url },
							},
						});
					}}
					multiple={false}
					render={({ open }) => (
						<>
							{image?.desktop ? <img src={image.desktop} /> : ""}
							<Button variant="primary" onClick={open}>
								{image?.desktop === null
									? "Select media file"
									: "Select Desktop image"}
							</Button>
						</>
					)}
				/>
			</PanelRow>

			<Divider />

			<PanelRow>
				<h2>Image Mobile</h2>
			</PanelRow>
			<PanelRow>
				<MediaUpload
					allowedTypes={["image"]}
					onSelect={(media) => {
						setAttributes({
							[name]: {
								...attribute,
								image: { ...image, mobile: media.url },
							},
						});
					}}
					multiple={false}
					render={({ open }) => (
						<>
							{image?.mobile ? <img src={image.mobile} /> : ""}
							<Button variant="primary" onClick={open}>
								{image?.mobile === null
									? "Select media file"
									: "Select Mobile image"}
							</Button>
						</>
					)}
				/>
			</PanelRow>
		</>
	) : (
		<></>
	);
};

const RenderVideoControl = ({
	render,
	video,
	setAttributes,
	attribute,
	name,
}) => {
	return render ? (
		<>
			<PanelRow>
				<h2>Video</h2>
			</PanelRow>
			<PanelRow>
				<MediaUpload
					allowedTypes={["video"]}
					onSelect={(media) => {
						setAttributes({ [name]: { ...attribute, video: media.url } });
					}}
					multiple={false}
					render={({ open }) => (
						<>
							{video ? (
								<video
									playsInline
									autoPlay
									loop
									muted
									data-type="autoplay"
									src={video}
								></video>
							) : (
								""
							)}
							<Button variant="primary" onClick={open}>
								{video === null ? "Select media file" : "Select video"}
							</Button>
						</>
					)}
				/>
			</PanelRow>
		</>
	) : (
		<></>
	);
};

export default {
	control: ({ title, name, attribute, setAttributes }) => {
		const [isVideo, setIsVideo] = useState(false);
		const image = attribute?.image || null;
		const video = attribute?.video || null;

		useEffect(() => {
			video ? setIsVideo(true) : setIsVideo(false);
		}, []);

		return (
			<PanelBody title={`${title} Responsive Image`}>
				<PanelRow>
					<ToggleControl
						label={isVideo ? "Video" : "Photo"}
						help="Select Media Type"
						checked={isVideo}
						onChange={(newValue) => {
							setIsVideo(newValue);
							setAttributes({ [name]: {} });
						}}
					/>
				</PanelRow>

				<RenderImageControl
					image={image}
					render={!isVideo}
					attribute={attribute}
					setAttributes={setAttributes}
					name={name}
				/>

				<RenderVideoControl
					video={video}
					render={isVideo}
					attribute={attribute}
					setAttributes={setAttributes}
					name={name}
				/>
			</PanelBody>
		);
	},
	view: ({ attribute }) => {
		const image = attribute?.image || null;
		const video = attribute?.video || null;
		return (
			<>
				{image?.desktop || image?.mobile ? (
					<picture className="img-box" style={{ with: "100%", height: "100%" }}>
						{image.desktop && (
							<source media="(min-width: 961px)" srcSet={image.desktop} />
						)}
						{image.mobile && (
							<source media="(min-width: 961px)" srcSet={image.mobile} />
						)}
						<img src={image.mobile || image.desktop} alt="" />
					</picture>
				) : (
					""
				)}
				{video && (
					<video playsInline autoPlay loop muted data-type="autoplay" src={video}></video>
				)}
			</>
		);
	},
	edit: ({ attribute }) => {
		const image = attribute?.image || null;
		const video = attribute?.video || null;
		return (
			<>
				{image?.desktop || image?.mobile ? (
					<picture className="img-box" style={{ with: "100%", height: "100%" }}>
						{image.desktop && (
							<source media="(min-width: 961px)" srcSet={image.desktop} />
						)}
						{image.mobile && (
							<source media="(min-width: 961px)" srcSet={image.mobile} />
						)}
						<img src={image.mobile || image.desktop} alt="" />
					</picture>
				) : (
					""
				)}
				{video && (
					<video playsInline autoPlay loop muted data-type="autoplay" src={video}></video>
				)}
			</>
		);
	},
};
