/**
 * External dependencies
 */
import memoize from 'memize';
import times from 'lodash/times';

/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { ButtonBlockAppender, RichText, InnerBlocks, useBlockProps } from '@wordpress/block-editor';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * Allowed blocks and template constant is passed to InnerBlocks precisely as specified here.
 * The contents of the array should never change. HEAD
 * The array should contain the name of each block that is allowed.
 * In standout block, the only block we allow is 'core/list'.
 *
 * @constant
 * @type {string[]}
 */
const ALLOWED_BLOCKS = [ 'nihi/floorplans-gallery-item' ];

/**
 * Returns the layouts configuration for a given number of accordion items.
 *
 * @param {number} count Number of accordion items.
 * @return {Object[]} Columns layout configuration.
 */
const getCount = memoize( ( count ) => {
	return times( count, () => [ 'nihi/floorplans-gallery-item' ] );
} );

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, clientId }) {
	const { title, count } = attributes;

	return (
		<section { ...useBlockProps() } className="container floorplans-gallery" data-comp="floorplans-gallery">
			<header className="row">
				<article className="col-12-11 offset-left-12-1"> 
					<RichText
						tagName="h2"
						value={title}
						onChange={(newValue) => setAttributes({ title: newValue })}
						placeholder="Title..."
					/>
				</article>
			</header>
			<section className="row">
				<div className="col-12-11 offset-left-12-1 splide-wrapper">
					<section className="splide" aria-label="Floorplans gallery slider">
						<div className="splide__track">
							<div className="splide__list">
								<InnerBlocks
									__experimentalCaptureToolbars={ true }
									allowedBlocks={ ALLOWED_BLOCKS }
									template={ getCount( count ) }
								/>
								<ButtonBlockAppender rootClientId={ clientId } />
							</div>
						</div>
					</section>
				</div>
			</section>
			<div className="floorplans-gallery__arrows">
				<button className="floorplans-gallery__arrow--prev" data-direction="next"><span>Prev</span> <i className="icon-arrow-left" role="img" aria-label="Icon arrow left"></i></button>
				<button className="floorplans-gallery__arrow--next" data-direction="prev"><span>Next</span> <i className="icon-arrow-right" role="img" aria-label="Icon arrow right"></i></button>
			</div>
		</section>
	);
}
