/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { InspectorControls, MediaUpload, MediaUploadCheck, RichText, useBlockProps } from '@wordpress/block-editor';
import { Button, Panel, PanelBody, PanelRow, TextControl } from '@wordpress/components';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes }) {
	const { sliders } = attributes;

	const removeSlide = (i) => {
		const newSlide = [...sliders];
		newSlide.splice(i, 1);
		setAttributes({
			sliders: newSlide,
		});
	};
	const addSlide = (val) => {
		const newSlide = [...sliders];
		const slide = {
			title: 'New slide',
			status: '',
			asset: {
				type: false
			},
			cta: {
				link: '',
				text: ''
			}
		}
		newSlide.push(slide);
		setAttributes({
			sliders: newSlide,
		});
	};

	const onSlideChange = (field, value, index, subfield) => {
		console.log('Before Update:', sliders);
		const updatedSliders = [...sliders];
		if(field == 'cta'){
			updatedSliders[index][field][subfield] = value;
		} else {
			updatedSliders[index][field] = value;
		}
		setAttributes({ sliders: updatedSliders });
		console.log('After Update:', updatedSliders);
	};

	return (
		<section {...useBlockProps()} className="container gallery-of-three-videos block-editor-block-list__block wp-block" data-comp="gallery-of-three-videos">
			<InspectorControls key='setting'>
				<Panel className="components-panel--custom">
					<React.Fragment key=".0">
						{sliders.map((value, i) => (
							<PanelBody key={'item-' + i} title={`Slide ${i}`}>
								<PanelRow>
									<TextControl
										label="Title"
										value={value.title}
										onChange={(val) =>
											onSlideChange('title', val, i)
										}
									/>
									<TextControl
										label="Status"
										value={value.status}
										onChange={(val) =>
											onSlideChange('status', val, i)
										}
									/>
									<MediaUploadCheck>
										<MediaUpload
											onSelect={(media) => onSlideChange('asset', media, i)}
											allowedTypes={['image', 'video']}
											value={value?.asset?.id || ''}
											render={({ open }) => (
												value.asset.type ? (
													value.asset.type == 'video' ? (
														<img
														src={value.asset.thumb.src}
														alt=""
														onClick={open}
														></img>
													) : (
														<img
															src={value.asset.sizes.thumbnail.url}
															alt=""
															onClick={open}
														/>
													)
												) : (
												<Button variant="secundary" onClick={open} className='bttn--secondary'>Select Image or video</Button>
												)
											)}
										/>
									</MediaUploadCheck>
									<TextControl
										label="CTA Link"
										value={Object.keys(value.cta).length > 0 ? value.cta.link : ''}
										onChange={(val) =>
											onSlideChange('cta', val, i, 'link')
										}
									/>
									<TextControl
										label="CTA Text"
										value={Object.keys(value.cta).length > 0 ? value.cta.text : ''}
										onChange={(val) =>
											onSlideChange('cta', val, i, 'text')
										}
									/>
									<Button
										variant='secondary'
										onClick={(e) => {
											removeSlide(i);
										}}
										className='bttn--danger'
									>
										Remove slide
									</Button>
								</PanelRow>
							</PanelBody>
						))}
						<Button
							variant='primary'
							onClick={(e) => {
								addSlide('new slide');
							}}
						>
							Add Slide
						</Button>
					</React.Fragment>
				</Panel>
			</InspectorControls>
			<div className="row stretch">
				<div className="col-12-12 gallery-of-three-videos__col no-mobile">
					<div className="gallery-of-three-videos__text">
						<RichText
							tagName="h3"
							value={attributes.title}
							onChange={(newHeading) => setAttributes({ title: newHeading })}
							placeholder="commited to"
						/>
						<RichText
							tagName="h2"
							value={attributes.subtitle}
							onChange={(newHeading) => setAttributes({ subtitle: newHeading })}
							placeholder="a shared future"
						/>
					</div>
					<div className="gallery-of-three-videos__action">
						<RichText
							tagName="h3"
							value={attributes.listTitle}
							onChange={(newHeading) => setAttributes({ listTitle: newHeading })}
							placeholder="PROPERTIES"
						/>
						<ul>
							{sliders.map((value, i) => (
								<li key={i} data-state={i === 0 ? 'active' : ''} data-index={i}>
									{value.title} <span>{value.status}</span>
								</li>
							))}
						</ul>
					</div>
				</div>
			</div>
			<div className="outer-videos">
				<section className="splide" aria-label="Gallery of three videos slide">
					<div className="splide__track">
						<div className="splide__list">
							{sliders.map((value, i) => (
								<a key={i} className="splide__slide img-box" href={value.cta.link}>
									<figcaption>
										{value.title && (<h3>{value.title}</h3>)}
										{value.status && (<h3>{value.status}</h3>)}
										{value.cta.text && (<button>{value.cta.text}</button>)}
									</figcaption>
									{value.asset.type === 'video' ? (
										<video playsInline autoPlay loop muted data-type="autoplay" src={value.asset.url} />
									) : (
										<img src={value.asset.url} alt={value.asset.title} />
									)}
								</a>
							))}
						</div>
					</div>
				</section>
			</div>
		</section>
	);
}
