/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText } from '@wordpress/block-editor';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save(props) {
    const { attributes } = props;

    return (
        <section {...useBlockProps.save()} className="container gallery-of-three-videos" data-comp="gallery-of-three-videos">
            <div className="row stretch">
                <div className="col-12-12 gallery-of-three-videos__col">
                    <div className="gallery-of-three-videos__text">
                        {attributes.title && (<RichText.Content tagName="h3" value={attributes.title} />)}
                        {attributes.subtitle && (<RichText.Content tagName="h2" value={attributes.subtitle} />)}
                    </div>
                    <div className="gallery-of-three-videos__action no-mobile">
                        {attributes.listTitle && (<h3>{attributes.listTitle}</h3>)}
                        <ul>
                            {attributes.sliders.map((value, i) => (
                                <li key={i} data-state={i === 0 ? 'active' : ''} data-index={i}>
                                    {value.title} <span>{value.status}</span>
                                </li>
                            ))}
                        </ul>
                    </div>
                </div>
            </div>
            <div className="outer-videos" style={`--subtitle-length: ${attributes.subtitle.length}`}>
                <section className="splide" aria-label="Gallery of three videos slide">
                    <div className="splide__track">
                        <div className="splide__list">
                            {attributes.sliders.map((value, i) => (
                                value.cta && value.cta.link ? (
                                    <a key={i} className={`splide__slide img-box has-link`} href={value.cta.link} target={value.cta.openInNewTab ? '_blank' : '_self'} data-text={value.cta.text} rel="noopener" data-element="video-container">
                                        <figcaption>
                                            {value.title && (<h3>{value.title}</h3>)}
                                            {value.status && (<h4>{value.status}</h4>)}
                                            {value.cta.text && (<button>{value.cta.text}</button>)}
                                        </figcaption>
                                        {value.asset.type === 'video' ? (
                                            <>
                                                <video playsInline autoPlay loop muted data-type="autoplay" src={value.asset.url} />
                                                <div className="control-videox mobile" data-comp="control-video-mobile">
                                                    <button className="control-video__button">Sound on</button>
                                                </div>
                                            </>
                                        ) : (
                                            <img src={value.asset.url} alt={value.asset.title} />
                                        )}
                                    </a>
                                ) : (
                                    <div key={i} className="splide__slide img-box" data-element="video-container">
                                        <figcaption>
                                            {value.title && (<h3>{value.title}</h3>)}
                                            {value.status && (<h4>{value.status}</h4>)}
                                            {value.cta.text && (<button>{value.cta.text}</button>)}
                                        </figcaption>
                                        {value.asset.type === 'video' ? (
                                            <>
                                                <video playsInline autoPlay loop muted data-type="autoplay" src={value.asset.url} />
                                                <div className="control-videox mobile" data-comp="control-video-mobile">
                                                    <button className="control-video__button">Sound on</button>
                                                </div>
                                            </>
                                        ) : (
                                            <img src={value.asset.url} alt={value.asset.title} />
                                        )}
                                    </div>
                                )
                            ))}
                        </div>
                    </div>
                </section>
            </div>
        </section>
    );
}
