/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText, MediaUpload, InspectorControls} from '@wordpress/block-editor';
import { Button, CheckboxControl, TextControl, PanelBody } from '@wordpress/components';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit(props) {
	const { attributes, setAttributes } = props;

	const imageHoverLink = (value, type) => {
		const { link, title, target } = attributes.imageLink;
	  
		const updatedImageLink = {
		  ...attributes.imageLink,
		  [type === 'link' ? 'link' : type === 'title' ? 'title' : 'target']: value,
		};
	  
		setAttributes({ imageLink: updatedImageLink });
	};

	const videoHoverLink = (value, type) => {
		const { link, title, target } = attributes.videoLink;
	  
		const updatedVideoLink = {
		  ...attributes.videoLink,
		  [type === 'link' ? 'link' : type === 'title' ? 'title' : 'target']: value,
		};
	  
		setAttributes({ videoLink: updatedVideoLink });
	};

	const getImageHoverLink = () => {
		const link = attributes.imageLink.link && attributes.imageLink.link.trim() !== '' ? attributes.imageLink.link : '';
		
		if(link == ''){
			return;
		}

		const target = attributes.imageLink.target ? '_blank' : '_self';
		const title = attributes.imageLink.title && attributes.imageLink.title.trim() !== '' ? attributes.imageLink.title : link;


		const anchorTag = `<a href="${link}" className="button" target="${target}">${title}</a>`;

		const anchorElement = new DOMParser().parseFromString(anchorTag, 'text/html');
  		const anchorContent = anchorElement.body.firstChild;

		return (
			<AnchorElement
			  href={anchorContent.href}
			  className={anchorContent.className}
			  target={anchorContent.target}
			>
			  {anchorContent.textContent}
			</AnchorElement>
		);
	}

	const getVideoHoverLink = () => {
		const link = attributes.videoLink.link && attributes.videoLink.link.trim() !== '' ? attributes.videoLink.link : '';
		
		if(link == ''){
			return;
		}

		const target = attributes.videoLink.target ? '_blank' : '_self';
		const title = attributes.videoLink.title && attributes.videoLink.title.trim() !== '' ? attributes.videoLink.title : link;


		const anchorTag = `<a href="${link}" className="button" target="${target}">${title}</a>`;

		const anchorElement = new DOMParser().parseFromString(anchorTag, 'text/html');
  		const anchorContent = anchorElement.body.firstChild;

		return (
			<AnchorElement
			  href={anchorContent.href}
			  className={anchorContent.className}
			  target={anchorContent.target}
			>
			  {anchorContent.textContent}
			</AnchorElement>
		);
	}

	const AnchorElement = ({ href, className, target, children }) => (
		<a href={href} className={className} target={target}>
		  {children}
		</a>
	);

	return (
		<section { ...useBlockProps() } className="container have-fun-categories-crosselling gutenberg-component">
			<InspectorControls>
				<PanelBody title="Image hover link">
					<TextControl
						help="Only accepts URL"
						label="Link"
						onChange={(newValue) => imageHoverLink(newValue, 'link')}
						value={ attributes.imageLink.link }
					/>
					<TextControl
						help="Title to show"
						label="Title"
						onChange={(newValue) => imageHoverLink(newValue, 'title')}
						value={ attributes.imageLink.title }
					/>
					<CheckboxControl
						label="Open in new tab?"
						help="Link is open in new tab"
						checked={ attributes.imageLink.target }
						onChange={(newValue) => imageHoverLink(newValue, 'target')}
					/>
				</PanelBody>
				<PanelBody title="Video hover link">
					<TextControl
						help="Only accepts URL"
						label="Link"
						onChange={(newValue) => videoHoverLink(newValue, 'link')}
						value={ attributes.videoLink.link }
					/>
					<TextControl
						help="Title to show"
						label="Title"
						onChange={(newValue) => videoHoverLink(newValue, 'title')}
						value={ attributes.videoLink.title }
					/>
					<CheckboxControl
						label="Open in new tab?"
						help="Link is open in new tab"
						checked={ attributes.videoLink.target }
						onChange={(newValue) => videoHoverLink(newValue, 'target')}
					/>
				</PanelBody>
			</InspectorControls>
			<header className="row">
				<div className="col-12-12 center">
					<hr />
					<RichText
                        tagName="h3"
                        value={attributes.heading}
                        onChange={(newHeading) => setAttributes({ heading: newHeading })}
                        placeholder="H3 TITLE"
                    />
				</div>
			</header>
			<div className="row">
				<figure className="col-12-6 img-box">
					<MediaUpload
						onSelect={(media) => setAttributes({ imageUrl: media.url })}
						allowedTypes={['image']}
						value={attributes.imageUrl}
						render={({ open }) => (
							attributes.imageUrl ? (
								<img
									src={attributes.imageUrl}
									alt=""
									onClick={open}
								/>
							) : (
								<Button variant="primary" onClick={open}>Select Image</Button>
							)
						)}
					/>
					<figcaption>
						{ getImageHoverLink() }
					</figcaption>
				</figure>
				<figure className="col-12-6 img-box">
					<MediaUpload
						onSelect={(media) => setAttributes({ videoUrl: media.url })}
						allowedTypes={['video']}
						value={attributes.videoUrl}
						render={({ open }) => (
							attributes.videoUrl ? (
								<video playsInline autoPlay loop muted data-type="autoplay" onClick={open} src={attributes.videoUrl}></video>
							) : (
								<Button variant="primary" onClick={open}>Select Video</Button>
							)
						)}
					/>
					<figcaption>
						{ getVideoHoverLink() }
					</figcaption>
				</figure>
			</div>
		</section>
	);
}
