/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText } from "@wordpress/block-editor";

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save(props) {
	const { attributes } = props;

	const heading = attributes.heading;
	const imageUrl = attributes.imageUrl;
	const imageLink = attributes.imageLink;
	const videoUrl = attributes.videoUrl;
	const videoLink = attributes.videoLink;

	return (
		<section
			{...useBlockProps.save()}
			className="container have-fun-categories-crosselling gutenberg-component"
		>
			<header className="row">
				<div className="col-12-12 center">
					<hr />
					{heading && (
						<RichText.Content
							tagName="h3"
							value={heading}
							className="heading"
						/>
					)}
				</div>
			</header>
			<div className="row">
				<figure className="col-12-6 img-box">
					<img src={imageUrl} alt="" />
					<figcaption>
						{imageLink && imageLink.link && imageLink.link.trim() !== "" ? (
							<a
								href={imageLink.link}
								className="button"
								target={imageLink.target ? "_blank" : "_self"}
							>
								{imageLink.title && imageLink.title.trim() !== ""
									? imageLink.title
									: imageLink.link}
							</a>
						) : null}
					</figcaption>
				</figure>
				<figure className="col-12-6 img-box">
					<video
						playsInline
						autoPlay
						loop
						muted
						data-type="autoplay"
						src={videoUrl}
					></video>
					<figcaption>
						{videoLink && videoLink.link && videoLink.link.trim() !== "" ? (
							<a
								href={videoLink.link}
								className="button"
								target={videoLink.target ? "_blank" : "_self"}
							>
								{videoLink.title && videoLink.title.trim() !== ""
									? videoLink.title
									: videoLink.link}
							</a>
						) : null}
					</figcaption>
				</figure>
			</div>
		</section>
	);
}
