/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from "@wordpress/i18n";

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	InspectorControls,
	useBlockProps,
	InnerBlocks,
} from "@wordpress/block-editor";
import { Panel, PanelBody, PanelRow, TextControl } from "@wordpress/components";

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import "./editor.scss";
import preview from './assets/text-in-the-bottom-header.jpg';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes }) {
	const { secID, buttonText, buttonTextHover } = attributes;

	return (
		<section
			{...useBlockProps()}
			className="container text-in-bottom-header block-editor-block-list__block wp-block"
			data-comp="text-in-bottom-header"
			data-target-id={secID}
			data-preview="true"
		>

			<figure className="preview">
				<img src={preview} alt="" />
			</figure>

			<InspectorControls key="setting">
				<Panel className="components-panel--custom">
					<PanelBody key="0" title="Settings">
						<PanelRow>
							<TextControl
								label="Section ID"
								value={secID}
								onChange={(newVal) => setAttributes({ secID: newVal })}
							/>
							<TextControl
								label="Button Text"
								value={buttonText}
								onChange={(newVal) => setAttributes({ buttonText: newVal })}
							/>
							<TextControl
								label="Button hover Text"
								value={buttonTextHover}
								onChange={(newVal) =>
									setAttributes({ buttonTextHover: newVal })
								}
							/>
						</PanelRow>
					</PanelBody>
				</Panel>
			</InspectorControls>
			<div className="row">
				<a className="col-12-12 center">
					{buttonText && <p className="rest">{buttonText}</p>}
					{buttonTextHover && <p className="hover">{buttonTextHover}</p>}
					<i className="icon-arrow-down"></i>
				</a>
			</div>
			<header className="col-12-12 center">
				<InnerBlocks
					allowedBlocks={[
						"core/separator",
						"core/heading",
						"core/paragraph",
						"core/html",
						"core/list",
						"core/quote",
						"core/pullquote",
						"core/table",
						"core/details",
					]}
					template={[
						["core/separator", { className: "big" }],
						["core/separator", { className: "small" }],
						["core/heading", { level: 2, placeholder: "Title" }],
						["core/heading", { level: 3, placeholder: "Subtitle" }],
						["core/paragraph", { placeholder: "Ingrese su párrafo aquí" }],
						["core/list", { ordered: true, placeholder: "Ordered List" }],
						["core/quote", { value: "Quote Text" }],
						["core/pullquote", { value: "Pullquote Text" }],
						["core/table", { value: "Table" }],
						["core/details", { value: "Details" }],
					]}
				/>

			</header>
		</section>
	);
}
